/*
* voltage_ctrl.c- Sigmastar
*
* Copyright (c) [2019~2020] SigmaStar Technology.
*
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License version 2 for more details.
*
*/
#if defined(__KERNEL__)
#include <linux/kernel.h>
#include <linux/of.h>
#include <linux/device.h>
#include <linux/module.h>
#endif
#include "mdrv_gpio_io.h"
#include "registers.h"
#include "ms_platform.h"
#include "voltage_ctrl.h"
#include "voltage_ctrl_demander.h"
#include "voltage_internal.h"
#include "voltage_os_impl.h"
#include "voltage_internal.h"
#include "cam_os_wrapper.h"
#include "cam_inter_os.h"

u8 enable_scaling_voltage = 0;

unsigned int gpio_vid_width = 0;
unsigned int *gpio_vid_pins = NULL;
unsigned int *gpio_vid_voltages = NULL;

int _gCurrentVoltageCore = 0;

int _gVoltageDemanderRequestValue[VOLTAGE_DEMANDER_MAX] = {0};
const char *_gVoltageDemanderName[] = {
    FOREACH_DEMANDER(GENERATE_STRING)
};
#ifdef CONFIG_SS_VOLTAGE_CTRL_WITH_OSC
struct voltage_range {
       int TT;
       int LV;
};
static struct voltage_range _gVoltageDemanderRange[VOLTAGE_DEMANDER_MAX] = {{0}};
#endif
int gVoltageCoreGpioInited = 0;
CamOsMutex_t core_voltage_mutex = {{0}};

u8 EnableScalingVoltageGet(void)
{
    return enable_scaling_voltage;
}

void EnableScalingVoltageSet(u8 en)
{
    CamOsMutexLock(&core_voltage_mutex);
    enable_scaling_voltage = en;
    CamOsMutexUnlock(&core_voltage_mutex);
}

int DemanderRequestValueGet(int idx)
{
    int ret = 0;

    if (idx < VOLTAGE_DEMANDER_MAX)
        ret = _gVoltageDemanderRequestValue[idx];

    return ret;
}

char *DemanderNameGet(int idx)
{
    char *ret = NULL;

    if (idx < VOLTAGE_DEMANDER_MAX)
        ret = (char *)_gVoltageDemanderName[idx];

    return ret;
}
 
#ifndef CONFIG_SS_DUALOS
static int check_voltage_valid(int mV)
{
    int voltage_lv_cnt = 0;
    int vcore_total_lv = 1 << gpio_vid_width;

    // Check core voltage setting(mV) valid or not.
    for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
    {
        if (mV == gpio_vid_voltages[voltage_lv_cnt])
        {
            break;
        }
    }

    if (voltage_lv_cnt == vcore_total_lv)  // If core voltage setting(mV) is invalid, Try to find ceiling setting.
    {
        for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
        {
            if (mV < gpio_vid_voltages[voltage_lv_cnt])
            {
                VOLCTRL_ERR("[Core Voltage] %s: Not support %dmV, use %dmV\n",
                            __FUNCTION__, mV, gpio_vid_voltages[voltage_lv_cnt]);
                mV = gpio_vid_voltages[voltage_lv_cnt];
                break;
            }
        }

        if (voltage_lv_cnt == vcore_total_lv)  // If no ceiling setting, use highest setting.
        {
            voltage_lv_cnt--;
            VOLCTRL_ERR("[Core Voltage] %s: Not support %dmV, use %dmV\n",
                        __FUNCTION__, mV, gpio_vid_voltages[voltage_lv_cnt]);
            mV = gpio_vid_voltages[voltage_lv_cnt];
        }
    }

    return mV;
}
#endif

void sync_core_voltage(void)
{
#ifndef CONFIG_SS_DUALOS
    int voltage_lv_cnt = 0;
    int gpio_lv_cnt = 0;
    int i = 0;
    int vcore_max = 0;
    int vcore_total_lv = 1 << gpio_vid_width;

    CamOsMutexLock(&core_voltage_mutex);

    if(gVoltageCoreGpioInited && gpio_vid_width && EnableScalingVoltageGet())
    {
        for (i=0; i<VOLTAGE_DEMANDER_MAX; i++)
        {
            vcore_max = max(vcore_max, _gVoltageDemanderRequestValue[i]);
        }
        vcore_max = check_voltage_valid(vcore_max);
        VOLCTRL_DBG("[Core Voltage] %s: maximum request is %dmV\n", __FUNCTION__, vcore_max);

        if(gpio_vid_width > 0)
        {
            for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
            {
                if (vcore_max == gpio_vid_voltages[voltage_lv_cnt])
                {
                    for (gpio_lv_cnt=0; gpio_lv_cnt<gpio_vid_width; gpio_lv_cnt++)
                    {
                        if ((voltage_lv_cnt >> gpio_lv_cnt) & 0x1)
                        {
                            camdriver_gpio_direction_output(NULL, gpio_vid_pins[gpio_lv_cnt], 1);
                            VOLCTRL_DBG("[Core Voltage] %s: gpio%d set high\n", __FUNCTION__, gpio_vid_pins[gpio_lv_cnt]);
                        }
                        else
                        {
                            camdriver_gpio_direction_output(NULL, gpio_vid_pins[gpio_lv_cnt], 0);
                            VOLCTRL_DBG("[Core Voltage] %s: gpio%d set low\n", __FUNCTION__, gpio_vid_pins[gpio_lv_cnt]);
                        }
                    }

                    _gCurrentVoltageCore = gpio_vid_voltages[voltage_lv_cnt];
                    break;
                }
            }
        }
    }
    else
    {
        VOLCTRL_DBG("[Core Voltage] %s: voltage scaling not enable\n", __FUNCTION__);
    }

    VOLCTRL_DBG("[Core Voltage] %s: current core voltage %dmV\n", __FUNCTION__, _gCurrentVoltageCore);

    CamOsMutexUnlock(&core_voltage_mutex);
#endif
}

#ifdef CONFIG_SS_VOLTAGE_CTRL_WITH_OSC
void voltage_level_reconfig (VOLTAGE_DEMANDER_E demander, int voltage)
{

     if (demander == VOLTAGE_DEMANDER_TEMPERATURE)
        return;

     _gVoltageDemanderRange[demander].TT = voltage;

     if (demander == VOLTAGE_DEMANDER_USER  || (gpio_vid_width == 1))
     {
         _gVoltageDemanderRange[demander].LV = voltage;
     }
     else
     {
         if (voltage == VOLTAGE_CORE_1000) {
             _gVoltageDemanderRange[demander].LV = VOLTAGE_CORE_950;
         }
         else {
             _gVoltageDemanderRange[demander].LV = VOLTAGE_CORE_900;
         }
     }
}

int sync_core_voltage_with_OSC_and_TEMP( int useTT)
{
    int i;
    int update = 0;

    CamOsMutexLock(&core_voltage_mutex);

    for (i=0; i<VOLTAGE_DEMANDER_MAX; i++)
    {
        if (_gVoltageDemanderRequestValue[i] != 0) {
            if (useTT) {
                if (_gVoltageDemanderRequestValue[i] != _gVoltageDemanderRange[i].TT) {
                    _gVoltageDemanderRequestValue[i] = _gVoltageDemanderRange[i].TT;
                    update = 1;
                }
            }
            else {
                if (_gVoltageDemanderRequestValue[i] != _gVoltageDemanderRange[i].LV) {
                    _gVoltageDemanderRequestValue[i] = _gVoltageDemanderRange[i].LV;
                    update = 1;
                }
            }
        }
    }
    CamOsMutexUnlock(&core_voltage_mutex);

    return update;
}
#endif

void set_core_voltage(VOLTAGE_DEMANDER_E demander, int mV)
{
#ifndef CONFIG_SS_DUALOS
    int voltage_lv_cnt = 0;
    int gpio_lv_cnt = 0;
    int i = 0;
    int vcore_max = 0;
    int vcore_total_lv = 1 << gpio_vid_width;
#endif

    if (demander >= VOLTAGE_DEMANDER_MAX)
    {
        VOLCTRL_ERR("[Core Voltage] %s: demander number out of range (%d)\n", __FUNCTION__, demander);
        return;
    }

#ifdef CONFIG_SS_DUALOS
    if (EnableScalingVoltageGet())
    {
        CamInterOsSignal(INTEROS_SC_L2R_CORE_VOLTAGE_SET, (u32)demander, (u32)mV, 0);
    }
#else
    CamOsMutexLock(&core_voltage_mutex);

#ifdef CONFIG_SS_VOLTAGE_CTRL_WITH_OSC
    voltage_level_reconfig(demander, mV);
#endif

    _gVoltageDemanderRequestValue[demander] = mV;
    VOLCTRL_DBG("[Core Voltage] %s: %s request %dmV\n", __FUNCTION__, _gVoltageDemanderName[demander], mV);

    if(gVoltageCoreGpioInited && gpio_vid_width && EnableScalingVoltageGet())
    {
        for (i=0; i<VOLTAGE_DEMANDER_MAX; i++)
        {
            vcore_max = max(vcore_max, _gVoltageDemanderRequestValue[i]);
        }
        vcore_max = check_voltage_valid(vcore_max);
        VOLCTRL_DBG("[Core Voltage] %s: maximum request is %dmV\n", __FUNCTION__, vcore_max);

        if(gpio_vid_width > 0)
        {
            for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
            {
                if (vcore_max == gpio_vid_voltages[voltage_lv_cnt])
                {
                    for (gpio_lv_cnt=0; gpio_lv_cnt<gpio_vid_width; gpio_lv_cnt++)
                    {
                        if ((voltage_lv_cnt >> gpio_lv_cnt) & 0x1)
                        {
                            camdriver_gpio_direction_output(NULL, gpio_vid_pins[gpio_lv_cnt], 1);
                            VOLCTRL_DBG("[Core Voltage] %s: gpio%d set high\n", __FUNCTION__, gpio_vid_pins[gpio_lv_cnt]);
                        }
                        else
                        {
                            camdriver_gpio_direction_output(NULL, gpio_vid_pins[gpio_lv_cnt], 0);
                            VOLCTRL_DBG("[Core Voltage] %s: gpio%d set low\n", __FUNCTION__, gpio_vid_pins[gpio_lv_cnt]);
                        }
                    }

                    _gCurrentVoltageCore = gpio_vid_voltages[voltage_lv_cnt];
                    break;
                }
            }
        }
    }
    else
    {
        VOLCTRL_DBG("[Core Voltage] %s: voltage scaling not enable\n", __FUNCTION__);
    }

    VOLCTRL_DBG("[Core Voltage] %s: current core voltage %dmV\n", __FUNCTION__, _gCurrentVoltageCore);

    CamOsMutexUnlock(&core_voltage_mutex);
#endif
}

int get_core_voltage(void)
{
#ifdef CONFIG_SS_DUALOS
    return CamInterOsSignal(INTEROS_SC_L2R_CORE_VOLTAGE_GET, 0, 0, 0);
#else
    return _gCurrentVoltageCore;
#endif
}

static int core_voltage_get_gpio(void)
{
    unsigned int i;
    char name[10];
    int ret;

    CamOsMutexLock(&core_voltage_mutex);

    if (0 != GetDeviceTreeVidWidth(&gpio_vid_width) || gpio_vid_width < 1)
        goto core_voltage_get_gpio_err;

    gpio_vid_pins = CamOsMemCalloc(sizeof(unsigned int) * gpio_vid_width, 1);
    if (!gpio_vid_pins)
        goto core_voltage_get_gpio_err;

    gpio_vid_voltages = CamOsMemCalloc(sizeof(unsigned int) * (1 << gpio_vid_width), 1);
    if (!gpio_vid_voltages)
        goto core_voltage_get_gpio_err;

    if (gpio_vid_width != GetDeviceTreeVidGpiosArray(gpio_vid_pins, gpio_vid_width))
        goto core_voltage_get_gpio_err;
    if ((1 << gpio_vid_width) != GetDeviceTreeVidVoltagesArray(gpio_vid_voltages, (1 << gpio_vid_width)))
        goto core_voltage_get_gpio_err;
    for(i = 0; i < gpio_vid_width; i++)
    {
        sprintf(name, "vid%d", i);
        ret = camdriver_gpio_request(NULL, gpio_vid_pins[i]);
        if (ret)
            goto core_voltage_get_gpio_err;
        //gpio_export(gpio_vid_pins[i], 0);
    }

    gVoltageCoreGpioInited = 1;

    CamOsMutexUnlock(&core_voltage_mutex);
    return 0;

core_voltage_get_gpio_err:
    gpio_vid_width = 0;
    if (gpio_vid_pins)
    {
        CamOsMemRelease(gpio_vid_pins);
        gpio_vid_pins = NULL;
    }
    if (gpio_vid_voltages)
    {
        CamOsMemRelease(gpio_vid_voltages);
        gpio_vid_voltages = NULL;
    }

    CamOsMutexUnlock(&core_voltage_mutex);
    return -1;
}

int core_voltage_available(unsigned int **voltages, unsigned int *num)
{
    if (voltages && num)
    {
        *num = 1 << gpio_vid_width;
        *voltages = gpio_vid_voltages;
        return 0;
    }
    else
    {
        return -1;
    }
}

int core_voltage_pin(unsigned int **pins, unsigned int *num)
{
    if (pins && num)
    {
        *num = gpio_vid_width;
        *pins = gpio_vid_pins;
        return 0;
    }
    else
    {
        return -1;
    }
}

int voltage_control_init(void)
{
    int ret = 0;
    int i;

    CamOsMutexInit(&core_voltage_mutex);

    ret = core_voltage_get_gpio();
    if (ret) {
        CamOsPrintf(KERN_ERR "Failed to get gpio for voltage control!! %d\n",ret);
        goto voltage_control_init_err;
    }

    VOLCTRL_DBG("[Core Voltage] %s: register sub system\n", __FUNCTION__);

    voltage_control_interface_init();

    // Initialize voltage request for specific IP by chip
    voltage_request_chip();

    // Enable core voltage scaling
    VOLCTRL_DBG("[Core Voltage] %s: turn-on core voltage scaling\n", __FUNCTION__);
    EnableScalingVoltageSet(1);
    sync_core_voltage();

    return 0;

voltage_control_init_err:
    for(i = 0; i < gpio_vid_width; i++)
    {
        camdriver_gpio_free(NULL, gpio_vid_pins[i]);
    }

    gpio_vid_width = 0;
    if (gpio_vid_pins)
    {
        CamOsMemRelease(gpio_vid_pins);
        gpio_vid_pins = NULL;
    }
    if (gpio_vid_voltages)
    {
        CamOsMemRelease(gpio_vid_voltages);
        gpio_vid_voltages = NULL;
    }

    return -1;
}

#if defined(__KERNEL__)
device_initcall(voltage_control_init);
#elif defined(CAM_OS_RTK)
rtos_device_initcall(voltage_control_init);
#endif
